' TpiNetClient.cs
'
' This file implements a Form-derived class that is also
' a remote TPI.NET client. It demonstrates how to connect
' to a remote TLA and subscribe to events raised by TPI.NET.
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

Imports Tektronix.LogicAnalyzer.TpiNet
Imports System
Imports System.Windows.Forms

Public Class Form1
    Inherits System.Windows.Forms.Form

    ' A reference to the top of the TPI.NET object hierarchy
    ' in the remote TLA application
    Dim m_system As ITlaSystem

    ' Keep track of how many acquisitions we have run.
    Dim m_startCount As Integer
    Dim m_endCount As Integer

    ' Event shims that are required to subscribe to
    ' events raised by the remote TLA application.
    Dim m_runStartHandler As EventHandler
    Dim m_runCompleteHandler As EventHandler

    ' The constructor connects to the remote TLA application and 
    ' subscribes to two of the ITlaRunControl events.
    Public Sub New()
        MyBase.New()

        'This call is required by the Windows Form Designer.
        InitializeComponent()

        Dim version As String = Nothing

        ' The assembly TlaNetInterfaces, which this project references,
        ' contains a utility class named RemoteTlaSystem. It has a
        ' Connect() static method which is used to set up a connection
        ' to a remote TLA. To connect, an application configuration file
        ' named ITlaSystemRemoteClient.config is used. This file contains
        ' the information needed to configure this application to
        ' communicate with a TLA running on the local host. The
        ' configuration file can be modified to connect to a TLA
        ' on a network.
        ' NOTE: For the call to Connect() to succeed, the file
        ' ITlaSystemRemoteClient.config must be in the same directory as
        ' the application executable.
        Try
            m_system = RemoteTlaSystem.Connect("ITlaSystemRemoteClient.config")
            version = m_system.SWVersion
        Catch
            ' Failed to connect.
            MessageBox.Show("Could not connect to the TLA application. Make sure the TLA application is running before starting the client application. Also make sure the configuration file is located in the executable directory.")
            m_system = Nothing
        End Try

        If m_system Is Nothing Then
            versionLabel.Text = "Could not connect to TLA application"
            runButton.Enabled = False
            runButton.Text = "Cannot run TLA"
        Else
            ' Connection was successful. Now subscribe to TPI.NET events.
            SubscribeToTpiNetEvents()

            versionLabel.Text = String.Format("TLA Software Version is {0}", version)

            ' Local events don't need shims.
            AddHandler runButton.Click, AddressOf Me.OnRunButton
        End If

    End Sub

    'Add TPI.NET event subscriptions.
    Private Sub SubscribeToTpiNetEvents()
        If Not m_system Is Nothing Then

            ' Subscribe to ITlaRunControl.RunStarted.
            m_runStartHandler = EventRemoter.Create(New EventHandler(AddressOf OnRunStarted))
            AddHandler m_system.RunControl.RunStarted, m_runStartHandler

            ' Subscribe to ITlaRunControl.RunCompleted.
            m_runCompleteHandler = EventRemoter.Create(New EventHandler(AddressOf OnRunCompleted))
            AddHandler m_system.RunControl.RunCompleted, m_runCompleteHandler
        End If
    End Sub

    ' Removes TPI.NET event subscriptions when the user exits the application. 
    ' TPI.NET clients should always clean up their subscriptions before exiting.
    Private Sub UnsubscribeFromTpiNetEvents()
        If Not m_system Is Nothing Then

            ' Unsubscribe from the run events.
            RemoveHandler m_system.RunControl.RunStarted, m_runStartHandler
            RemoveHandler m_system.RunControl.RunCompleted, m_runCompleteHandler

            m_runStartHandler = Nothing
            m_runCompleteHandler = Nothing
        End If
    End Sub

#Region " Windows Form Designer generated code "

    'Form overrides dispose to clean up the component list.
    Protected Overloads Overrides Sub Dispose(ByVal disposing As Boolean)
        If disposing Then
            UnsubscribeFromTpiNetEvents()

            If Not (components Is Nothing) Then
                components.Dispose()
                components = Nothing
            End If
        End If
        MyBase.Dispose(disposing)
    End Sub

    'Required by the Windows Form Designer
    Private components As System.ComponentModel.IContainer

    'NOTE: The following procedure is required by the Windows Form Designer
    'It can be modified using the Windows Form Designer.  
    'Do not modify it using the code editor.
    Friend WithEvents versionLabel As System.Windows.Forms.Label
    Friend WithEvents runButton As System.Windows.Forms.Button
    Friend WithEvents Label1 As System.Windows.Forms.Label
    Friend WithEvents eventLabel As System.Windows.Forms.Label
    <System.Diagnostics.DebuggerStepThrough()> Private Sub InitializeComponent()
        Me.versionLabel = New System.Windows.Forms.Label()
        Me.runButton = New System.Windows.Forms.Button()
        Me.Label1 = New System.Windows.Forms.Label()
        Me.eventLabel = New System.Windows.Forms.Label()
        Me.SuspendLayout()
        '
        'versionLabel
        '
        Me.versionLabel.Location = New System.Drawing.Point(16, 16)
        Me.versionLabel.Name = "versionLabel"
        Me.versionLabel.Size = New System.Drawing.Size(256, 24)
        Me.versionLabel.TabIndex = 0
        Me.versionLabel.Text = "Label1"
        '
        'runButton
        '
        Me.runButton.Location = New System.Drawing.Point(40, 56)
        Me.runButton.Name = "runButton"
        Me.runButton.Size = New System.Drawing.Size(224, 32)
        Me.runButton.TabIndex = 1
        Me.runButton.Text = "Run the TLA"
        '
        'Label1
        '
        Me.Label1.Location = New System.Drawing.Point(24, 112)
        Me.Label1.Name = "Label1"
        Me.Label1.Size = New System.Drawing.Size(264, 24)
        Me.Label1.TabIndex = 2
        Me.Label1.Text = "Most Recent Run Event:"
        '
        'eventLabel
        '
        Me.eventLabel.Location = New System.Drawing.Point(24, 152)
        Me.eventLabel.Name = "eventLabel"
        Me.eventLabel.Size = New System.Drawing.Size(264, 24)
        Me.eventLabel.TabIndex = 3
        Me.eventLabel.Text = "None."
        '
        'Form1
        '
        Me.AutoScaleBaseSize = New System.Drawing.Size(6, 15)
        Me.ClientSize = New System.Drawing.Size(312, 216)
        Me.Controls.AddRange(New System.Windows.Forms.Control() {Me.eventLabel, Me.Label1, Me.runButton, Me.versionLabel})
        Me.Name = "Form1"
        Me.Text = "VB Remote TPI.NET Client"
        Me.ResumeLayout(False)

    End Sub

#End Region


    ' Event handler for the ITlaRunControl.RunStarted event.
    Private Sub OnRunStarted(ByVal sender As Object, ByVal ea As EventArgs)
        ' Asynchronous events from the TLA Application come in on worker threads.
        ' In order to update the GUI controls, we must be on the UI thread.
        If (InvokeRequired) Then
            ' Make a call to the UI thread to output the results.
            Dim args() As Object = {sender, ea}
            BeginInvoke(New EventHandler(AddressOf OnRunStarted), args)

        Else
            ' We're already on the UI thread.  Output the results directly.
            m_startCount += 1
            eventLabel.Text = String.Format("Acquisition {0} started.", m_startCount)
            runButton.Enabled = False
        End If
    End Sub

    'Event handler for the ITlaRunControl.RunCompleted event.
    Private Sub OnRunCompleted(ByVal sender As Object, ByVal ea As EventArgs)
        ' Asynchronous events from the TLA Application come in on worker threads.
        ' In order to update the GUI controls, we must be on the UI thread.
        If (InvokeRequired) Then
            ' Make a call to the UI thread to output the results.
            Dim args() As Object = {sender, ea}
            BeginInvoke(New EventHandler(AddressOf OnRunCompleted), args)

        Else
            ' We're already on the UI thread.  Output the results directly.
            m_endCount += 1
            eventLabel.Text = String.Format("Acquisition {0} completed.", m_endCount)
            runButton.Enabled = True
        End If
    End Sub


    ' Starts a TLA acquistion when the user clicks the 
    ' "Run the TLA" button.
    Private Sub OnRunButton(ByVal sender As Object, ByVal args As EventArgs)
        If Not m_system Is Nothing Then
            m_system.RunControl.Run()
        End If
    End Sub

End Class
